<?php

namespace Automattic\Jetpack\WP_JS_Data_Sync;

use Automattic\Jetpack\WP_JS_Data_Sync\Schema\Parser;

class DS_Utils {

	private static $mode;

	/**
	 * Is the current environment a development environment?
	 *
	 * @return bool
	 */
	public static function is_debug(): bool {
		return (
			( defined( 'DATASYNC_DEBUG' ) && DATASYNC_DEBUG )
			||
			self::$mode === 'debug'
			||
			self::$mode === 'debug-verbose'
		);
	}

	public static function is_verbose() {
		return (
			self::is_debug()
			&&
			(
				self::$mode === 'debug-verbose'
				||
				( defined( 'DATASYNC_VERBOSE' ) && DATASYNC_VERBOSE )
			)
		);
	}

	public static function set_mode( $mode ) {
		$valid_modes = array( 'debug', 'debug-verbose', null );

		if ( ! in_array( $mode, $valid_modes, true ) ) {
			self::$mode = null;
			return false;
		}

		self::$mode = $mode;
	}

	public static function debug_disable( $name ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['ds-debug-disable'] ) && ( is_admin() || ( defined( 'REST_REQUEST' ) && REST_REQUEST ) ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$values = explode( ',', sanitize_key( $_GET['ds-debug-disable'] ) );
			if ( ! $values || ! is_array( $values ) ) {
				return false;
			}
			if ( $values === array( 'all' ) ) {
				return true;
			}
			return in_array( $name, $values, true );
		}
		return false;
	}

	/**
	 * The schema generated by the parser can be very verbose and hard to read.
	 * This is a helper utility that converts the schema into a more human-readable format.
	 *
	 * @param Parser $parser The parser instance.
	 *                       This is the same parser instance that was used to generate the schema.
	 *                       It is used to convert the schema into a human-readable format.
	 * @param bool   $encode Whether to encode the schema as a JSON string.
	 *
	 * @return array|string
	 */
	public static function describe(
		Parser $parser,
		$encode = false
	) {
		// Process the top-level schema array
		$description = self::human_readable_schema( $parser->schema() );

		if ( $encode === true ) {
			return $description;
		}
		// Convert the processed schema to a JSON-like string
		return wp_json_encode( $description, JSON_PRETTY_PRINT );
	}

	/**
	 * Recursive function that will crawl through the schema,
	 * and call on itself to process each sub-item.
	 *
	 * @param mixed $item The item to process.
	 *
	 * @return mixed The processed item.
	 */
	private static function human_readable_schema(
		$item
	) {
		// If the item is an associative array with 'type' as a key, return its value directly
		if ( is_array( $item ) && isset( $item['type'] ) && count( $item ) === 1 ) {
			return $item['type'];
		}

		// If the item is an associative array with 'type' and 'value', process the value
		if ( isset( $item['type'], $item['value'] ) && is_array( $item ) ) {
			return self::human_readable_schema( $item['value'] );
		}

		// If the item is any other kind of array, process each sub-item
		if ( is_array( $item ) ) {
			$result = array();
			foreach ( $item as $key => $value ) {
				$result[ $key ] = self::human_readable_schema( $value );
			}
			return $result;
		}
		return $item;
	}
}
