<?php

/*
Plugin Name: Security Ninja
Plugin URI: https://wpsecurityninja.com/
Description: Check your site for <strong>security vulnerabilities</strong> and get precise suggestions for corrective actions on passwords, user accounts, file permissions, database security, version hiding, plugins, themes and other security aspects.
Author: WP Security Ninja
Version: 5.80
Author URI: https://wpsecurityninja.com/
* Text Domain: security-ninja
* Domain Path: /languages
*
Copyright 2011-2020 Web factory Ltd

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
This plugin uses the following 3rd party MIT licensed projects - Thank you for making other developer lives easier :-)

* UserAgentParser by Jesse G. Donat - https://github.com/donatj/PhpUserAgent

* Country flags Copyright (c) 2017 Go Squared Ltd. http://www.gosquared.com/ - https://github.com/gosquared/flags. MIT license.

*  10k-most-common.txt passwords https://github.com/danielmiessler/SecLists

* PHP malware scanner - https://github.com/scr34m/php-malware-scanner
This plugin works on a modified version of the excellent PHP malware scanner.
*/
// this is an include only WP file
if ( !defined( 'ABSPATH' ) ) {
    exit;
}

if ( function_exists( 'secnin_fs' ) ) {
    secnin_fs()->set_basename( false, __FILE__ );
} else {
    
    if ( !function_exists( 'secnin_fs' ) ) {
        // Create a helper function for easy SDK access.
        function secnin_fs()
        {
            global  $secnin_fs ;
            
            if ( !isset( $secnin_fs ) ) {
                // Activate multisite network integration.
                if ( !defined( 'WP_FS__PRODUCT_3690_MULTISITE' ) ) {
                    define( 'WP_FS__PRODUCT_3690_MULTISITE', true );
                }
                // Include Freemius SDK.
                require_once dirname( __FILE__ ) . '/freemius/start.php';
                $secnin_fs = fs_dynamic_init( array(
                    'id'              => '3690',
                    'slug'            => 'security-ninja',
                    'type'            => 'plugin',
                    'public_key'      => 'pk_f990ec18700a90c02db544f1aa986',
                    'is_premium'      => false,
                    'premium_suffix'  => '',
                    'has_addons'      => false,
                    'has_paid_plans'  => true,
                    'trial'           => array(
                    'days'               => 14,
                    'is_require_payment' => true,
                ),
                    'has_affiliation' => 'selected',
                    'menu'            => array(
                    'slug'    => 'wf-sn',
                    'support' => false,
                ),
                    'is_live'         => true,
                ) );
            }
            
            return $secnin_fs;
        }
        
        // Init Freemius.
        secnin_fs();
        // Signal that SDK was initiated.
        do_action( 'secnin_fs_loaded' );
    }
    
    // constants
    define( 'WF_SN_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
    define( 'WF_SN_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
    define( 'WF_SN_BASE_FILE', __FILE__ );
    define( 'WF_SN_RESULTS_KEY', 'wf_sn_results' );
    define( 'WF_SN_OPTIONS_KEY', 'wf_sn_options' );
    define( 'WF_SN_POINTERS_KEY', 'wf_sn_pointers' );
    define( 'WF_SN_MAX_EXEC_SEC', 200 );
    define( 'WF_SN_TEXT_DOMAIN', 'security-ninja' );
    define( 'WF_SN_INSTALLED_DB_KEY', 'wf_sn_installed_db_version' );
    define( 'WF_SN_REVIEW_NOTICE_KEY', 'wf_sn_review_notice' );
    define( 'WF_SN_ACTIVE_PLUGINS', 'wf_sn_active_plugins' );
    require WF_SN_PLUGIN_DIR . 'vendor/autoload.php';
    /**
     * Custom logo URL for Freemius dialogue
     * @return string Absolute URL
     */
    function secnin_fs_custom_icon()
    {
        return dirname( __FILE__ ) . '/images/sn-logo.svg';
    }
    
    secnin_fs()->add_filter( 'plugin_icon', 'secnin_fs_custom_icon' );
    require_once WF_SN_PLUGIN_DIR . 'modules/vulnerabilities/vulnerabilities.php';
    class wf_sn
    {
        static  $version = 0 ;
        static  $name = 'Security Ninja' ;
        static  $licensing_servers = array( 'https://license1.wpsecurityninja.com/', 'http://license2.wpsecurityninja.com/' ) ;
        static  $skip_tests = array() ;
        // init plugin
        static function init()
        {
            // SN requires WP v4.4
            
            if ( !version_compare( get_bloginfo( 'version' ), '4.4', '>=' ) ) {
                add_action( 'admin_notices', array( __CLASS__, 'min_version_error' ) );
                return;
            }
            
            // Set default wait until show review notice first time.
            $review = get_option( WF_SN_REVIEW_NOTICE_KEY );
            
            if ( !$review ) {
                $review = array(
                    'time'      => time() + WEEK_IN_SECONDS * 4,
                    'dismissed' => false,
                );
                update_option( WF_SN_REVIEW_NOTICE_KEY, $review, 'no' );
            }
            
            // Load security tests
            require_once WF_SN_PLUGIN_DIR . 'sn-tests.php';
            // does the user have enough privilages to use the plugin?
            
            if ( current_user_can( 'administrator' ) ) {
                // Adds helpscout permission to Freemius
                
                if ( function_exists( 'secnin_fs' ) ) {
                    secnin_fs()->add_filter( 'permission_list', array( __CLASS__, 'add_freemius_extra_permission' ) );
                    secnin_fs()->add_filter(
                        'show_admin_notice',
                        array( __CLASS__, 'do_filter_show_admin_notice' ),
                        10,
                        2
                    );
                    // Automatic license migration
                    add_action( 'admin_init', array( __CLASS__, 'secnin_fs_license_key_migration' ) );
                }
                
                // Returns tabs for admin page
                add_filter(
                    'sn_tabs',
                    array( __CLASS__, 'return_tabs' ),
                    5001,
                    2
                );
                // add menu item to tools
                add_action( 'admin_menu', array( __CLASS__, 'admin_menu' ) );
                // aditional links in plugin description
                add_filter( 'plugin_action_links_' . basename( dirname( __FILE__ ) ) . '/' . basename( __FILE__ ), array( __CLASS__, 'plugin_action_links' ), 99 );
                // Set to high so to overrule "change license"
                add_filter(
                    'plugin_row_meta',
                    array( __CLASS__, 'plugin_meta_links' ),
                    10,
                    2
                );
                // enqueue scripts
                add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_scripts' ) );
                // register settings key
                add_action( 'admin_init', array( __CLASS__, 'register_settings' ) );
                add_action( 'wp_ajax_sn_run_tests', array( __CLASS__, 'run_tests' ) );
                add_action( 'wp_ajax_sn_dismiss_pointer', array( __CLASS__, 'dismiss_pointer_ajax' ) );
                // warn if tests were not run
                add_action( 'admin_notices', array( __CLASS__, 'run_tests_warning' ) );
                add_action( 'admin_notices', array( __CLASS__, 'do_admin_notices' ) );
                // Ask for a review
                add_action( 'wp_ajax_wf_sn_dismiss_review', array( __CLASS__, 'wf_sn_dismiss_review' ) );
                // add_markup for UI overlay
                add_action( 'admin_footer', array( __CLASS__, 'admin_footer' ) );
            }
            
            // if is_admin
        }
        
        // init
        // Update dismissed notice
        static function wf_sn_dismiss_review()
        {
            $review = get_option( WF_SN_REVIEW_NOTICE_KEY );
            if ( !$review ) {
                $review = array();
            }
            $review['time'] = time() + WEEK_IN_SECONDS * 4;
            $review['dismissed'] = true;
            update_option( WF_SN_REVIEW_NOTICE_KEY, $review );
            die;
        }
        
        static function timerstart( $watchname )
        {
            set_transient( 'security_ninja_' . $watchname, microtime( true ), 60 * 60 * 1 );
        }
        
        static function timerstop( $watchname, $digits = 5 )
        {
            $return = round( microtime( true ) - get_transient( 'security_ninja_' . $watchname ), $digits );
            delete_transient( 'security_ninja_' . $watchname );
            return $return;
        }
        
        // *******
        static function do_admin_notices()
        {
            $is_sn_admin_page = self::is_plugin_page();
            if ( !$is_sn_admin_page ) {
                return;
            }
            $review = get_option( WF_SN_REVIEW_NOTICE_KEY );
            $time = time();
            $load = false;
            
            if ( !$review ) {
                $review = array(
                    'time'      => $time,
                    'dismissed' => false,
                );
                $load = true;
            } else {
                // Check if it has been dismissed or not.
                if ( isset( $review['dismissed'] ) && !$review['dismissed'] && (isset( $review['time'] ) && $review['time'] <= $time) ) {
                    $load = true;
                }
            }
            
            // Hvis vi skal vise den igen
            if ( isset( $review['time'] ) ) {
                if ( $time > $review['time'] ) {
                    // Vi kan godt vise den igen
                    $load = true;
                }
            }
            if ( !$load ) {
                return;
            }
            // Update the review option now.
            update_option( WF_SN_REVIEW_NOTICE_KEY, $review, 'no' );
            $current_user = wp_get_current_user();
            $fname = '';
            if ( !empty($current_user->user_firstname) ) {
                $fname = $current_user->user_firstname;
            }
            if ( function_exists( 'secnin_fs' ) ) {
                
                if ( secnin_fs()->is_registered() ) {
                    $get_user = secnin_fs()->get_user();
                    $fname = $get_user->first;
                }
            
            }
            // We have a candidate! Output a review message.
            //
            $timeused = __( 'a while', WF_SN_TEXT_DOMAIN );
            $options = self::get_options();
            if ( isset( $options['first_install'] ) ) {
                $timeused = human_time_diff( $options['first_install'], current_time( 'timestamp' ) );
            }
            ?>
		<div class="notice notice-info is-dismissible wfsn-review-notice">
			<p><?php 
            printf( __( 'Hey %s, I noticed you have been using Security Ninja for %s - that’s awesome!', WF_SN_TEXT_DOMAIN ), $fname, $timeused );
            ?></p>

			<p><?php 
            printf( __( 'Could you please do us a BIG favor and give it a 5-star rating on WordPress to help us spread the word?', WF_SN_TEXT_DOMAIN ), $fname, $timeused );
            ?></p>
			<p><?php 
            _e( 'Thank you :-)', WF_SN_TEXT_DOMAIN );
            ?></p>
			<p><strong>Lars Koudal,</br>wpsecurityninja.com</strong></p>
			<p><ul>
				<li><a href="https://wordpress.org/support/plugin/security-ninja/reviews/?filter=5#new-post" class="wfsn-dismiss-review-notice wfsn-reviewlink button-primary" target="_blank" rel="noopener"><?php 
            _e( 'Ok, you deserve it', WF_SN_TEXT_DOMAIN );
            ?></a></li>
				<li><span class="dashicons dashicons-calendar"></span><a href="#" class="wfsn-dismiss-review-notice" target="_blank" rel="noopener"><?php 
            _e( 'Nope, maybe later', WF_SN_TEXT_DOMAIN );
            ?></a></li>
				<li><span class="dashicons dashicons-smiley"></span><a href="#" class="wfsn-dismiss-review-notice" target="_blank" rel="noopener"><?php 
            _e( 'I already did', WF_SN_TEXT_DOMAIN );
            ?></a></li>
			</ul>
		</p>
		<p><small><?php 
            _e( 'This notice is shown every 30 days.', WF_SN_TEXT_DOMAIN );
            ?></small></p>
	</div>
	<?php 
        }
        
        // do_admin_notices
        //
        //
        //
        /**
         * Fetch plugin version from plugin PHP header
         * @return string Plugin version
         */
        static function get_plugin_version()
        {
            $plugin_data = get_file_data( __FILE__, array(
                'version' => 'Version',
            ), 'plugin' );
            wf_sn::$version = $plugin_data['version'];
            return $plugin_data['version'];
        }
        
        /**
         * Fetch plugin version from plugin PHP header - Free / Pro
         * @return string Plugin name
         */
        static function get_plugin_name()
        {
            $plugin_data = get_file_data( __FILE__, array(
                'name' => 'Plugin Name',
            ), 'plugin' );
            wf_sn::$name = $plugin_data['name'];
            return $plugin_data['name'];
        }
        
        /**
         * Checks for and migrates old license system to Freemius automatically.
         * @return void
         */
        static function secnin_fs_license_key_migration()
        {
            if ( !secnin_fs()->has_api_connectivity() || secnin_fs()->is_registered() ) {
                // No connectivity OR the user already opted-in to Freemius.
                return;
            }
            if ( 'pending' != get_option( 'secnin_fs_migrated2fs', 'pending' ) ) {
                return;
            }
            // Get the license key from the previous eCommerce platform's storage.
            $options = self::get_options();
            $license_key = $options['license_key'];
            if ( empty($license_key) ) {
                // No key to migrate.
                return;
            }
            if ( strlen( $license_key ) < 32 ) {
                // Pad license key with zeros at the end.
                $license_key = str_pad( $license_key, 32, '0' );
            }
            // Get the first 32 characters.
            $license_key = substr( $license_key, 0, 32 );
            try {
                $next_page = secnin_fs()->activate_migrated_license( $license_key );
            } catch ( Exception $e ) {
                update_option( 'secnin_fs_migrated2fs', 'unexpected_error' );
                return;
            }
            
            if ( secnin_fs()->can_use_premium_code() ) {
                update_option( 'secnin_fs_migrated2fs', 'done' );
                if ( is_string( $next_page ) ) {
                    fs_redirect( $next_page );
                }
            } else {
                update_option( 'secnin_fs_migrated2fs', 'failed' );
            }
        
        }
        
        // Snagged from database-optimizer.php
        static function num_format_size( $bytes )
        {
            
            if ( $bytes > 1073741824 ) {
                return number_format_i18n( $bytes / 1073741824, 2 ) . ' GB';
            } elseif ( $bytes > 1048576 ) {
                return number_format_i18n( $bytes / 1048576, 1 ) . ' MB';
            } elseif ( $bytes > 1024 ) {
                return number_format_i18n( $bytes / 1024, 1 ) . ' KB';
            } else {
                return number_format_i18n( $bytes, 0 ) . ' bytes';
            }
        
        }
        
        // format_size
        static function render_events_logger_page()
        {
            echo  '<div class="submit-test-container">' ;
            ?>
		<div class="fomcont">
			<h3><?php 
            _e( 'Events Logger', WF_SN_TEXT_DOMAIN );
            ?></h3>

			<img src="<?php 
            echo  WF_SN_PLUGIN_URL . '/images/event-log.jpg' ;
            ?>" alt="<?php 
            _e( 'Scan Core files of WordPress', WF_SN_TEXT_DOMAIN );
            ?>" class="tabimage">

			<p><?php 
            _e( "The Events Logger monitors, tracks and reports every change on your WordPress site, both in the admin and on the frontend.", WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( 'Simple audit logging - Keep an activity log of what happens on your website and help troubleshoot bugs.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( 'Know what happened on the site at any time, in the admin and on the frontend.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( 'Easily filter trough events.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( 'Know exactly when and how an action happened, and who did it.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( 'Receive email alerts for selected groups of events.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( "More than 50 events are instantly tracked with all details.", WF_SN_TEXT_DOMAIN );
            ?></p>

			<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_events_logger', '/events-logger/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>

		</div>

	</div>
	<?php 
        }
        
        /**
         * Returns full URL for country flag.
         * @param  boolean $country Country code. Expected Uppercase - check filename structure in folder
         * @return [type]  Full URL or false if not exists.
         */
        static function get_country_img( $country = false )
        {
            if ( !$country ) {
                return false;
            }
            $filepath = trailingslashit( WF_SN_PLUGIN_DIR ) . 'images/flags/' . $country . '.png';
            
            if ( file_exists( $filepath ) ) {
                $fileurl = trailingslashit( WF_SN_PLUGIN_URL ) . 'images/flags/' . $country . '.png';
                return $fileurl;
            }
            
            return false;
        }
        
        static function get_country_img_src( $country = false, $size = 32 )
        {
            if ( !$country ) {
                return false;
            }
            $return = false;
            $country_img_url = self::get_country_img( $country );
            if ( !isset( $geoip_countrylist ) ) {
                require WF_SN_PLUGIN_DIR . 'modules/cloud-firewall/class-sn-geoip-countrylist.php';
            }
            $countryName = '';
            if ( isset( $geoip_countrylist[$country] ) ) {
                $countryName = $geoip_countrylist[$country];
            }
            if ( $country_img_url ) {
                $return = '<img src="' . $country_img_url . '" width="' . $size . '" height="' . $size . '" class="countryimg" title="' . $countryName . '">';
            }
            return $return;
        }
        
        static function render_vuln_page()
        {
            global  $wpdb ;
            ?>
	<div class="submit-test-container">
		<div class="fomcont">
			<h3><?php 
            _e( 'Known vulnerabilities', WF_SN_TEXT_DOMAIN );
            ?></h3>

			<?php 
            // @todo - show when list last updated
            // @todo - create weekly list update
            // @todo - create manual update button
            // @todo - show vulnerabilities
            // CAN ONLY BE USED TEMP - copy the function or include differently?
            // require_once WF_SN_PLUGIN_DIR . 'modules/malware-scanner/malware-scanner.php';
            // require_once WF_SN_PLUGIN_DIR . 'modules/malware-scanner/class-pluginsintegrity.php';
            ?>

			<p><?php 
            _e( "Here we list known vulnerabilities in plugins or themes you have installed, also if there are known vulnerabilities in the WordPress version you have installed.", WF_SN_TEXT_DOMAIN );
            ?></p>



			<p><?php 
            _e( "This feature is free for all, but Pro users have a few more options available.", WF_SN_TEXT_DOMAIN );
            ?></p>

			<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_vulnerabilities', '/vulnerabilities/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>
		</div>
	</div>
	<?php 
        }
        
        static function render_databaseoptimizer_page()
        {
            global  $wpdb ;
            ?>
	<div class="submit-test-container">
		<div class="fomcont">
			<h3><?php 
            _e( 'Database Optimizer', WF_SN_TEXT_DOMAIN );
            ?></h3>

			<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_events_logger', '/database-optimizer/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>
			<p><?php 
            _e( "As you use WordPress and add more content to your site, it will inevitably lead to garbage data accumulation in your database. And while ten or a couple of hundred records wont slow your site down a couple of thousand might, and tens of thousand definitely will.", WF_SN_TEXT_DOMAIN );
            ?></p>
			<p><?php 
            _e( "Speed aside, some people just love a clean database :-)", WF_SN_TEXT_DOMAIN );
            ?></p>

		</div>
	</div>
	<?php 
        }
        
        // Renders the output for the cloud firewall module
        static function render_cloudfw_page()
        {
            echo  '<div class="submit-test-container">' ;
            ?>
	<div class="fomcont">
		<h3><?php 
            _e( 'Cloud Firewall', WF_SN_TEXT_DOMAIN );
            ?></h3>

		<img src="<?php 
            echo  WF_SN_PLUGIN_URL . '/images/firewall.jpg' ;
            ?>" alt="<?php 
            _e( 'Scan Core files of WordPress', WF_SN_TEXT_DOMAIN );
            ?>" class="tabimage">

		<p><?php 
            _e( 'The Cloud Firewall is a dynamic, continuously changing database of bad IP addresses updated every six hours. It contains roughly 600 million IPs that are known for distributing malware, performing brute force attacks on sites and doing other "bad" activities. The database is created by analyzing log files of millions of sites.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p><?php 
            _e( "By using the cloud firewall, you will be one step ahead of the bad guys. They won't be able to login to your site.", WF_SN_TEXT_DOMAIN );
            ?></p>

		<p><?php 
            _e( 'Block suspicious requests - Each pageview is checked and blocked if necessary.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p><?php 
            _e( 'Login Protection - Block repeated failed login attempts, prevent brute force login attacks.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p><?php 
            _e( 'Country Blocking - Prevent visits from any country from visiting.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p><?php 
            _e( 'Show a message to blocked visitors or redirect them to any other URL.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_events_logger', '/cloud-firewall/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>

	</div>
	<?php 
            echo  '</div>' ;
        }
        
        // Renders the output for the malware module
        static function render_malware_page()
        {
            // @todo - load later?
            if ( class_exists( 'wf_sn_ms' ) ) {
                $last_run = wf_sn_ms::get_results( 'last_run' );
            }
            echo  '<div class="submit-test-container">' ;
            ?>
	<div class="fomcont">
		<h3><?php 
            _e( 'Malware Scanner', WF_SN_TEXT_DOMAIN );
            ?></h3>

		<img src="<?php 
            echo  WF_SN_PLUGIN_URL . '/images/malware-scanner.jpg' ;
            ?>" alt="<?php 
            _e( 'Find malicious files in your WordPress site', WF_SN_TEXT_DOMAIN );
            ?>" class="tabimage">

		<p><?php 
            _e( 'Protecting yourself from hacking attempts is always the best choice, but no matter, if you have a software firewall, enabled and use secure passwords your website can be hacked.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p><?php 
            _e( 'Each public plugin from wordpress.org will be checked against a master checklist to see if any plugin files has been modified.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p><?php 
            _e( 'The rest of the website will be scanned for code commenly found in malicious scripts and specifically known attacks.', WF_SN_TEXT_DOMAIN );
            ?></p>

		<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_events_logger', '/malware-scanner/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>

	</div>
	<?php 
            echo  '</div>' ;
            echo  '<div id="sn_ms_results">' ;
            
            if ( isset( $last_run ) && $last_run ) {
                $results = self::run_scan( 'get_results' );
                echo  $results ;
            }
            
            echo  '</div>' ;
            ?>

	<?php 
        }
        
        // malware_page
        static function render_scheduled_scanner_page()
        {
            echo  '<div class="submit-test-container">' ;
            ?>
		<div class="fomcont">
			<h3><?php 
            _e( 'Scheduled Scanner', WF_SN_TEXT_DOMAIN );
            ?></h3>

			<img src="<?php 
            echo  WF_SN_PLUGIN_URL . '/images/scheduler.jpg' ;
            ?>" alt="<?php 
            _e( 'Scan Core files of WordPress', WF_SN_TEXT_DOMAIN );
            ?>" class="tabimage">


			<p><?php 
            _e( "The Scheduled Scanner gives you an additional peace of mind by automatically running Security Ninja and Core Scanner tests every day.", WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( "If any changes occur or your site gets hacked you'll immediately get notified via email.", WF_SN_TEXT_DOMAIN );
            ?></p>

			<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_scheduled_scanner', '/scheduled-scanner/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>
		</div>
	</div>
	<?php 
        }
        
        // Renders the output for the core scanner module
        static function render_core_page()
        {
            ?>
	<div class="submit-test-container">
		<div class="fomcont">
			<h3><?php 
            _e( 'Core Scanner', WF_SN_TEXT_DOMAIN );
            ?></h3>

			<img src="<?php 
            echo  WF_SN_PLUGIN_URL . '/images/core-scanner.jpg' ;
            ?>" alt="<?php 
            _e( 'Scan Core files of WordPress', WF_SN_TEXT_DOMAIN );
            ?>" class="tabimage">

			<p><?php 
            _e( 'The Core Scanner compares all your core WordPress files (over 1,200) with the secure master copy maintained by WordPress.org.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( 'With one click you will know if even a byte was changed in any file. If so, you can imediatelly recover the original version.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><?php 
            _e( 'This helps you find infected files that should be removed.', WF_SN_TEXT_DOMAIN );
            ?></p>

			<p><strong><?php 
            _e( 'Perfect for restoring hacked sites!', WF_SN_TEXT_DOMAIN );
            ?></strong></p>

			<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_events_logger', '/core-scanner/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>


		</div>
	</div>
	<?php 
        }
        
        // Renders the output for the whitelabel page
        static function render_whitelabel_page()
        {
            ?>
	<div class="submit-test-container">
		<div class="fomcont">
			<h3><?php 
            _e( 'Whitelabel', WF_SN_TEXT_DOMAIN );
            ?></h3>

			<img src="<?php 
            echo  WF_SN_PLUGIN_URL . '/images/whitelabel.jpg' ;
            ?>" alt="<?php 
            _e( 'Whitelabel your security work.', WF_SN_TEXT_DOMAIN );
            ?>" class="tabimage">

			<p class="fomlink"><a target="_blank" href="<?php 
            echo  wf_sn::generate_sn_web_link( 'tab_whitelabel', '/' ) ;
            ?>" class="button button-primary"><?php 
            _e( 'Learn more', WF_SN_TEXT_DOMAIN );
            ?></a></p>
			<p>Whitelabel allows you to hide the account and contact links in the menu. It also hides notifications made by the processing company.</p>
			<p>You can enter a new name for the plugin, as well as your company URL.</p>
			<p>Note that all help features are also removed, it is up to you to help your customers :-)</p>
			<p></p>
			<p>This feature is available for Pro users with 25+ site licenses.</p>

		</div>
	</div>
	<?php 
        }
        
        /**
         * Prepares the tabs for the plugin interface
         *
         * @author  Lars Koudal <me@larsik.com>
         *
         * @since 2.6
         *
         * @param array    $intabs  Array of tabs for plugin to be processed
         */
        static function return_tabs( $intabs )
        {
            $outtabs = $intabs;
            $core_tab = array(
                'id'       => 'sn_core',
                'class'    => 'profeature',
                'label'    => 'Core Scanner',
                'callback' => array( __CLASS__, 'render_core_page' ),
            );
            $malware_tab = array(
                'id'       => 'sn_malware',
                'class'    => 'profeature',
                'label'    => 'Malware',
                'callback' => array( __CLASS__, 'render_malware_page' ),
            );
            $cloudfw_tab = array(
                'id'       => 'sn_cf',
                'class'    => 'profeature',
                'label'    => 'Firewall',
                'callback' => array( __CLASS__, 'render_cloudfw_page' ),
            );
            $schedule_tab = array(
                'id'       => 'sn_schedule',
                'class'    => 'profeature',
                'label'    => __( 'Scheduler', WF_SN_TEXT_DOMAIN ),
                'callback' => array( __CLASS__, 'render_scheduled_scanner_page' ),
            );
            $logger_tab = array(
                'id'       => 'sn_logger',
                'class'    => 'profeature',
                'label'    => 'Event Log',
                'callback' => array( __CLASS__, 'render_events_logger_page' ),
            );
            $dboptimizer_tab = array(
                'id'       => 'sn_do',
                'class'    => 'profeature hidden',
                'label'    => 'Database',
                'callback' => array( __CLASS__, 'render_databaseoptimizer_page' ),
            );
            $whitelabel_tab = array(
                'id'       => 'sn_whitelabel',
                'class'    => 'profeature',
                'label'    => 'Whitelabel',
                'callback' => array( __CLASS__, 'render_whitelabel_page' ),
            );
            global  $secnin_fs ;
            if ( isset( $core_tab ) ) {
                $outtabs[] = $core_tab;
            }
            if ( isset( $cloudfw_tab ) ) {
                $outtabs[] = $cloudfw_tab;
            }
            if ( isset( $schedule_tab ) ) {
                $outtabs[] = $schedule_tab;
            }
            if ( isset( $malware_tab ) ) {
                $outtabs[] = $malware_tab;
            }
            if ( isset( $logger_tab ) ) {
                $outtabs[] = $logger_tab;
            }
            if ( isset( $dboptimizer_tab ) ) {
                $outtabs[] = $dboptimizer_tab;
            }
            if ( isset( $whitelabel_tab ) ) {
                $outtabs[] = $whitelabel_tab;
            }
            return $outtabs;
        }
        
        static function add_freemius_extra_permission( $permissions )
        {
            $permissions['helpscout'] = array(
                'icon-class' => 'dashicons dashicons-sos',
                'label'      => 'Help Scout',
                'desc'       => __( 'Rendering Help Scouts beacon for easy help and support', WF_SN_TEXT_DOMAIN ),
                'priority'   => 16,
            );
            $permissions['wpsnapi'] = array(
                'icon-class' => 'dashicons dashicons-sos',
                'label'      => 'Security Ninja API',
                'desc'       => __( 'Getting data from Security Ninja API servers.', WF_SN_TEXT_DOMAIN ),
                'priority'   => 17,
            );
            $permissions['newsletter'] = array(
                'icon-class' => 'dashicons dashicons-email-alt2',
                'label'      => 'Newsletter',
                'desc'       => __( 'Your email is added to our newsletter. Unsubscribe anytime.', WF_SN_TEXT_DOMAIN ),
                'priority'   => 18,
            );
            return $permissions;
        }
        
        /**
         * @param bool  $show
         * @param array $msg {
         *     @var string $message The actual message.
         *     @var string $title An optional message title.
         *     @var string $type The type of the message ('success', 'update', 'warning', 'promotion').
         *     @var string $id The unique identifier of the message.
         *     @var string $manager_id The unique identifier of the notices manager. For plugins it would be the plugin's slug, for themes - `<slug>-theme`.
         *     @var string $plugin The product's title.
         *     @var string $wp_user_id An optional WP user ID that this admin notice is for.
         * }
         * @return bool
         */
        // Filters out any Freemius admin notices
        static function do_filter_show_admin_notice( $show, $msg )
        {
            return $show;
        }
        
        // some things have to be loaded earlier
        static function plugins_loaded()
        {
            wf_sn::get_plugin_version();
            load_plugin_textdomain( WF_SN_TEXT_DOMAIN, false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
        }
        
        // plugins_loaded
        // add links to plugin's description in plugins table
        static function plugin_meta_links( $links, $file )
        {
            $support_link = '<a target="_blank" href="https://wpsecurityninja.com/help/" title="' . __( 'Get help or create a support ticket', WF_SN_TEXT_DOMAIN ) . '">' . __( 'Support', WF_SN_TEXT_DOMAIN ) . '</a>';
            if ( $file == plugin_basename( __FILE__ ) ) {
                $links[] = $support_link;
            }
            return $links;
        }
        
        // plugin_meta_links
        // add settings link to plugins page
        static function plugin_action_links( $links )
        {
            $settings_link = '<a href="tools.php?page=wf-sn" title="Security Ninja">' . __( 'Secure the site', WF_SN_TEXT_DOMAIN ) . '</a>';
            array_unshift( $links, $settings_link );
            return $links;
        }
        
        // plugin_action_links
        /**
         * Test if we are on one of the pages in this plugin
         * @return boolean True/false
         */
        static function is_plugin_page()
        {
            $current_screen = get_current_screen();
            
            if ( in_array( $current_screen->id, array( 'toplevel_page_wf-sn' ) ) or strpos( $current_screen->id, 'page_wf-sn-debug' ) !== false or strpos( $current_screen->id, 'page_wf-sn-tools' ) !== false ) {
                return true;
            } else {
                return false;
            }
        
        }
        
        // is_plugin_page
        // enqueue CSS and JS scripts on plugin's pages
        static function enqueue_scripts()
        {
            
            if ( wf_sn::is_plugin_page() ) {
                wp_enqueue_script( 'jquery-ui-tabs' );
                wp_enqueue_script(
                    'sn-jquery-plugins',
                    WF_SN_PLUGIN_URL . 'js/min/sn-jquery-plugins-min.js',
                    array( 'jquery' ),
                    wf_sn::$version,
                    true
                );
                wp_enqueue_style( 'wp-jquery-ui-dialog' );
                wp_enqueue_script( 'jquery-ui-dialog' );
                // Parsing data to sn-common.js via $cp_sn_data
                wp_register_script(
                    'sn-js',
                    WF_SN_PLUGIN_URL . 'js/min/sn-common-min.js',
                    array( 'jquery' ),
                    wf_sn::$version,
                    true
                );
                $cp_sn_data = array(
                    'load_helpscout' => 0,
                );
                // parsing data to sn-common.js
                // If we have Freemius installed and is registered
                if ( function_exists( 'secnin_fs' ) && secnin_fs()->is_registered() ) {
                    // currently no effect, beacon is not loaded via .js but bottom of the main page
                    $cp_sn_data['load_helpscout'] = 1;
                }
                wp_localize_script( 'sn-js', 'cp_sn_data', $cp_sn_data );
                wp_enqueue_script( 'sn-js' );
                $current_screen = get_current_screen();
                wp_enqueue_style( 'site-health' );
                wp_enqueue_script( 'site-health' );
                if ( $current_screen->id == 'security-ninja_page_wf-sn-debug' ) {
                }
                wp_enqueue_style(
                    'sn-css',
                    WF_SN_PLUGIN_URL . 'css/min/sn-style.css',
                    array(),
                    wf_sn::$version
                );
                // Removing scripts and styles from other plugins we know mess up the interface
                wp_dequeue_style( 'uiStyleSheet' );
                wp_dequeue_style( 'wpcufpnAdmin' );
                wp_dequeue_style( 'unifStyleSheet' );
                wp_dequeue_style( 'wpcufpn_codemirror' );
                wp_dequeue_style( 'wpcufpn_codemirrorTheme' );
                wp_dequeue_style( 'collapse-admin-css' );
                wp_dequeue_style( 'jquery-ui-css' );
                wp_dequeue_style( 'tribe-common-admin' );
                wp_dequeue_style( 'file-manager__jquery-ui-css' );
                wp_dequeue_style( 'file-manager__jquery-ui-css-theme' );
                wp_dequeue_style( 'wpmegmaps-jqueryui' );
                wp_dequeue_style( 'facebook-plugin-css' );
                wp_dequeue_style( 'facebook-tip-plugin-css' );
                wp_dequeue_style( 'facebook-member-plugin-css' );
                wp_dequeue_style( 'kc-testimonial-admin' );
                wp_dequeue_style( 'jquery-ui-style' );
            }
            
            // if SN page
            $pointers = get_option( WF_SN_POINTERS_KEY );
            
            if ( !empty($pointers) && !wf_sn::is_plugin_page() ) {
                wp_enqueue_script( 'wp-pointer' );
                wp_enqueue_script(
                    'wf-sn-pointers',
                    plugins_url( 'js/min/sn-pointers-min.js', __FILE__ ),
                    array( 'jquery' ),
                    wf_sn::$version,
                    true
                );
                wp_enqueue_style( 'wp-pointer' );
                wp_localize_script( 'wp-pointer', 'wf_sn_pointers', $pointers );
            }
            
            
            if ( wf_sn::is_plugin_page() || !empty($pointers) ) {
                $js_vars = array(
                    'sn_plugin_url'         => WF_SN_PLUGIN_URL,
                    'nonce_run_tests'       => wp_create_nonce( 'wf_sn_run_tests' ),
                    'nonce_refresh_update'  => wp_create_nonce( 'wf_sn_refresh_update' ),
                    'nonce_dismiss_pointer' => wp_create_nonce( 'wf_sn_dismiss_pointer' ),
                    'lc_version'            => wf_sn::$version,
                    'lc_site'               => get_home_url(),
                    'lc_ip'                 => $_SERVER['REMOTE_ADDR'],
                );
                wp_localize_script( 'jquery', 'wf_sn', $js_vars );
            }
        
        }
        
        // enqueue_scripts
        // permanently dismiss pointer
        static function dismiss_pointer_ajax()
        {
            check_ajax_referer( 'wf_sn_dismiss_pointer' );
            $pointers = get_option( WF_SN_POINTERS_KEY );
            $pointer = trim( $_POST['pointer'] );
            if ( empty($pointers) || empty($pointers[$pointer]) ) {
                wp_send_json_error();
            }
            unset( $pointers[$pointer] );
            update_option( WF_SN_POINTERS_KEY, $pointers );
            wp_send_json_success();
        }
        
        // dismiss_pointer_ajax
        // add entry to admin menu
        static function admin_menu()
        {
            $page_title = 'Security';
            $menu_title = 'Security Ninja';
            $capability = 'manage_options';
            $menu_slug = 'wf-sn';
            $icon_url = '';
            $position = null;
            $icon_url = wf_sn::get_icon_svg();
            add_menu_page(
                $page_title,
                $menu_title,
                $capability,
                $menu_slug,
                array( __CLASS__, 'main_page' ),
                $icon_url
            );
            global  $wp_version ;
            if ( version_compare( $wp_version, '5.2.0', '>=' ) ) {
                add_submenu_page(
                    'wf-sn',
                    'Debug',
                    'Debug',
                    'manage_options',
                    'wf-sn-debug',
                    array( __CLASS__, 'create_debug_page' )
                );
            }
        }
        
        // admin_menu
        static function do_filter_debug_information( $info )
        {
            $info['wp-paths-sizes']['label'] = 'Directories';
            unset( $info['wp-paths-sizes']['fields']['wordpress_size'] );
            unset( $info['wp-paths-sizes']['fields']['uploads_size'] );
            unset( $info['wp-paths-sizes']['fields']['themes_size'] );
            unset( $info['wp-paths-sizes']['fields']['plugins_size'] );
            unset( $info['wp-paths-sizes']['fields']['database_size'] );
            unset( $info['wp-paths-sizes']['fields']['total_size'] );
            unset( $info['wp-plugins-active']['fields']['Security Ninja'] );
            if ( class_exists( 'wf_sn_wl' ) ) {
                
                if ( wf_sn_wl::is_active() ) {
                    $pluginname = wf_sn_wl::get_new_name();
                    if ( isset( $info['wp-plugins-active']['fields'][$pluginname] ) ) {
                        unset( $info['wp-plugins-active']['fields'][$pluginname] );
                    }
                }
            
            }
            return $info;
        }
        
        static function create_debug_page()
        {
            ?>
		<div class="wrap">
			<?php 
            $topbar = '<img src="' . WF_SN_PLUGIN_URL . 'images/sn-logo.svg' . '" height="28" alt="' . __( 'Visit wpsecurityninja.com', WF_SN_TEXT_DOMAIN ) . '" class="logoleft"><h1>Security Ninja <span>v.' . self::get_plugin_version() . '</span></h1>';
            echo  $topbar ;
            if ( !class_exists( 'WP_Debug_Data' ) ) {
                require_once ABSPATH . 'wp-admin/includes/class-wp-debug-data.php';
            }
            WP_Debug_Data::check_for_updates();
            $info = WP_Debug_Data::debug_data();
            // Cleans up directory sizes and labels
            $info = self::do_filter_debug_information( $info );
            $exportdata = WP_Debug_Data::format( $info, 'debug' );
            ?>
			<h3><?php 
            _e( 'Debug Information', WF_SN_TEXT_DOMAIN );
            ?></h3>
			<p>This page contains details about your WordPress installation.</p>

			<div class="secnin_content_wrapper">

				<div class="secnin_content_cell">


					<div id="health-check-debug" class="health-check-accordion">

						<?php 
            $sizes_fields = array(
                'uploads_size',
                'themes_size',
                'plugins_size',
                'wordpress_size',
                'database_size',
                'total_size'
            );
            foreach ( $info as $section => $details ) {
                if ( !isset( $details['fields'] ) || empty($details['fields']) ) {
                    continue;
                }
                ?>
							<h3 class="health-check-accordion-heading">
								<button aria-expanded="false" class="health-check-accordion-trigger" aria-controls="health-check-accordion-block-<?php 
                echo  esc_attr( $section ) ;
                ?>" type="button">
									<span class="title">
										<?php 
                echo  esc_html( $details['label'] ) ;
                ?>
										<?php 
                if ( isset( $details['show_count'] ) && $details['show_count'] ) {
                    printf( '(%d)', count( $details['fields'] ) );
                }
                ?>
									</span>
									<?php 
                if ( 'wp-paths-sizes' === $section ) {
                    ?>
										<span class="health-check-wp-paths-sizes spinner"></span>
										<?php 
                }
                ?>
									<span class="icon"></span>
								</button>
							</h3>

							<div id="health-check-accordion-block-<?php 
                echo  esc_attr( $section ) ;
                ?>" class="health-check-accordion-panel" hidden="hidden">
								<?php 
                if ( isset( $details['description'] ) && !empty($details['description']) ) {
                    printf( '<p>%s</p>', $details['description'] );
                }
                ?>
								<table class="widefat striped health-check-table" role="presentation">
									<tbody>
										<?php 
                foreach ( $details['fields'] as $field_name => $field ) {
                    
                    if ( is_array( $field['value'] ) ) {
                        $values = '<ul>';
                        foreach ( $field['value'] as $name => $value ) {
                            $values .= sprintf( '<li>%s: %s</li>', esc_html( $name ), esc_html( $value ) );
                        }
                        $values .= '</ul>';
                    } else {
                        $values = esc_html( $field['value'] );
                    }
                    
                    
                    if ( in_array( $field_name, $sizes_fields, true ) ) {
                        printf(
                            '<tr><td>%s</td><td class="%s">%s</td></tr>',
                            esc_html( $field['label'] ),
                            esc_attr( $field_name ),
                            $values
                        );
                    } else {
                        printf( '<tr><td>%s</td><td>%s</td></tr>', esc_html( $field['label'] ), $values );
                    }
                
                }
                ?>
									</tbody>
								</table>
							</div>
						<?php 
            }
            ?>
					</div><!-- #health-check-debug -->

					<h3><?php 
            _e( 'Debug Data', WF_SN_TEXT_DOMAIN );
            ?></h3>
					<p><?php 
            _e( 'If support asks you for more details about your website to help debugging, here it is in easy to copy and paste format.', WF_SN_TEXT_DOMAIN );
            ?></p>

					<div class="site-health-copy-buttons">
						<div class="copy-button-wrapper">
							<button type="button" class="button copy-button" data-clipboard-text="<?php 
            echo  esc_attr( WP_Debug_Data::format( $info, 'debug' ) ) ;
            ?>">
								<?php 
            _e( 'Copy site info to clipboard', WF_SN_TEXT_DOMAIN );
            ?>
							</button>
							<span class="success" aria-hidden="true"><?php 
            _e( 'Copied!', WF_SN_TEXT_DOMAIN );
            ?></span>
						</div>
					</div>

				</div><!-- .secnin_content_cell -->

				<?php 
            include_once 'misc/sidebar-help.php';
            ?>
			</div><!-- .secnin_content_wrapper -->
		</div>
		<?php 
        }
        
        // display warning if test were never run
        static function run_tests_warning()
        {
            if ( !wf_sn::is_plugin_page() ) {
                return;
            }
            $tests = get_option( WF_SN_RESULTS_KEY, array() );
            
            if ( !empty($tests['last_run']) && current_time( 'timestamp' ) - DAY_IN_SECONDS * 30 > $tests['last_run'] ) {
                ?>
			<div class="notice notice-error"><p>
				<?php 
                _e( "Tests were not run for more than 30 days! It's advisable to run them once in a while. Click 'Analyze Site' to run them now and analyze your site for security vulnerabilities.", WF_SN_TEXT_DOMAIN );
                ?>
			</p></div>
			<?php 
            }
            
            
            if ( empty($tests['last_run']) ) {
                ?>
			<div class="notice notice-warning"><p>
				<?php 
                _e( "You have not run the Security Tests - Get started on the 'Security Tests' tab.", WF_SN_TEXT_DOMAIN );
                ?>
			</p></div>
			<?php 
            }
        
        }
        
        // run_tests_warning
        // add_settings_error
        static function add_settings_error( $message, $type = 'error', $code = 'wf_sn' )
        {
            global  $wp_settings_errors ;
            $wp_settings_errors[] = array(
                'setting' => WF_SN_OPTIONS_KEY,
                'code'    => $code,
                'message' => $message,
                'type'    => $type,
            );
            set_transient( 'settings_errors', $wp_settings_errors );
        }
        
        // add_settings_error
        // display warning if test were never run
        static function min_version_error()
        {
            echo  '<div class="notice notice-error"><p>This plugin requires WordPress version 4.4 or higher to function properly. You\'re using WordPress version ' . get_bloginfo( 'version' ) . '. Please <a href="' . admin_url( 'update-core.php' ) . '" title="Update WP core">update</a>.</p></div>' ;
            // i8n
        }
        
        // min_version_error
        // add markup for UI overlay
        static function admin_footer()
        {
            
            if ( wf_sn::is_plugin_page() ) {
                echo  '<div id="sn_overlay"><div class="sn-overlay-wrapper">' ;
                echo  '<div class="inner">' ;
                // Outer
                echo  '<div class="wf-sn-overlay-outer">' ;
                // Content @todo
                echo  '<div class="wf-sn-overlay-content">' ;
                echo  '<div id="sn-site-scan" style="display: none;">' ;
                echo  '<h3>' . __( 'Analyzing your site.', WF_SN_TEXT_DOMAIN ) . '</h3>' ;
                echo  '<p id="wf-sn-last-action"></p>' ;
                echo  '<p id="wf-sn-last-msg"></p>' ;
                echo  '<p><label id="counterminutes">00</label>:<label id="counterseconds">00</label></p>' ;
                echo  '</div>' ;
                do_action( 'sn_overlay_content' );
                ?>
			<div class="loader"><img src="<?php 
                echo  esc_url( get_admin_url() . 'images/spinner-2x.gif' ) ;
                ?>" height="20" width="20" /></div>
			<?php 
                echo  '<p><a id="abort-scan" href="#" class="button button-secondary button">' . __( 'Stop', WF_SN_TEXT_DOMAIN ) . '</a></p>' ;
                do_action( 'sn_overlay_content_after' );
                echo  '</div>' ;
                // wf-sn-overlay-content
                echo  '</div></div></div></div>' ;
                echo  '<div id="test-details-dialog" style="display: none;" title="Test details"><p>' . __( 'Please wait.', WF_SN_TEXT_DOMAIN ) . '</p></div>' ;
                echo  '<div id="sn_tests_descriptions" style="display: none;">' ;
                require_once WF_SN_PLUGIN_DIR . 'sn-tests-description.php';
                echo  '</div>' ;
            }
            
            // if is_plugin_page
        }
        
        // admin_footer
        // return default options
        static function default_options()
        {
            $defaults = array(
                'license_key'     => '',
                'license_active'  => false,
                'license_expires' => '',
                'license_type'    => '',
                'license_hide'    => false,
            );
            return $defaults;
        }
        
        // default_settings
        // get plugin's options
        static function get_options()
        {
            $options = get_option( WF_SN_OPTIONS_KEY, array() );
            if ( !is_array( $options ) ) {
                $options = array();
            }
            $options = array_merge( self::default_options(), $options );
            return $options;
        }
        
        // get_options
        // all settings are saved in one option
        static function register_settings()
        {
            register_setting( WF_SN_OPTIONS_KEY, WF_SN_OPTIONS_KEY, array( __CLASS__, 'sanitize_settings' ) );
            if ( isset( $_POST['_wpnonce'] ) ) {
                $nonce = $_POST['_wpnonce'];
            }
            # we do not want to redirect everyone
            $redirect_user = false;
            if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( $nonce, 'wf-sn-install-routines' ) ) {
                $redirect_user = true;
            }
            
            if ( $redirect_user ) {
                // Set to false per default, so isset check not needed.
                if ( !isset( $_POST['_wp_http_referer'] ) ) {
                    $_POST['_wp_http_referer'] = wp_login_url();
                }
                $url = sanitize_text_field( wp_unslash( $_POST['_wp_http_referer'] ) );
                wp_safe_redirect( urldecode( $url ) );
                exit;
            }
        
        }
        
        // register_settings
        /**
         * Returns icon in SVG format
         *
         * Thanks Yoast for code.
         *
         * @param bool $base64      Return SVG in base64 or not
         * @return string
         */
        static function get_icon_svg( $base64 = true )
        {
            $svg = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 500 500">
		<g fill="#82878c">
		<path d="M171.117 262.277c14.583-.142 25.832 20.664 25.921 35.25.094 15.265-11.418 37.682-26.678 37.227-14.687-.438-23.797-22.605-23.494-37.296.295-14.24 10.095-35.044 24.25-35.181zM322.387 263.03c14.584-.142 25.832 20.664 25.922 35.25.093 15.265-11.419 37.681-26.679 37.227-14.686-.438-23.797-22.606-23.493-37.296.294-14.24 10.094-35.044 24.25-35.182z"/>
		<path d="M331.348 26.203c0-.107 98.038-7.914 98.038-7.914s-9.219 91.716-10.104 96.592c1.277-3.3 22.717-46.002 22.818-46.002.105 0 53.047 69.799 53.047 69.799l-46.63 42.993c26.6 30.762 41.632 67.951 41.724 107.653.239 103.748-110.253 191.827-245.68 191.091-130.352-.706-239.977-86.977-240.475-188.91-.5-102.38 105.089-191.741 239.663-192.095 38.677-.1 74.34 6.068 105.82 17.154-3.241-16.067-18.22-90.265-18.22-90.36zm-85.421 157.959c-74.098-1.337-161.3 41.627-161.054 105.87.247 63.88 87.825 103.981 160.683 104.125 78.85.154 164.156-41.58 163.722-106.614-.428-64.436-86.566-101.996-163.351-103.381z"/>
		</g>
		</svg>';
            if ( $base64 ) {
                return 'data:image/svg+xml;base64,' . base64_encode( $svg );
            }
            return $svg;
        }
        
        // Sanitize settings on save
        static function sanitize_settings( $new_values )
        {
            $old_options = self::get_options();
            return array_merge( $old_options, $new_values );
        }
        
        // sanitize_settings
        /**
         * Helper function to generate tagged links
         * @param  string $placement [description]
         * @param  string $page      [description]
         * @param  array  $params    [description]
         * @return string            Full URL with utm_ parameters added
         */
        static function generate_sn_web_link( $placement = '', $page = '/', $params = array() )
        {
            $base_url = 'https://wpsecurityninja.com';
            if ( '/' != $page ) {
                $page = '/' . trim( $page, '/' ) . '/';
            }
            $utm_source = 'security_ninja_free';
            $parts = array_merge( array(
                'utm_source'   => $utm_source,
                'utm_medium'   => 'plugin',
                'utm_content'  => $placement,
                'utm_campaign' => 'security_ninja_v' . self::$version,
            ), $params );
            $out = $base_url . $page . '?' . http_build_query( $parts, '', '&amp;' );
            return $out;
        }
        
        // whole options page
        static function main_page()
        {
            // redundant but doesn't hurt
            if ( !current_user_can( 'administrator' ) ) {
                wp_die( __( 'You do not have sufficient permissions to access this page.', WF_SN_TEXT_DOMAIN ) );
            }
            $options = self::get_options();
            global  $secnin_fs ;
            settings_errors();
            $tabs = array();
            $tabs[] = array(
                'id'       => 'sn_tests',
                'class'    => '',
                'label'    => __( 'Security Tests', WF_SN_TEXT_DOMAIN ),
                'callback' => array( __CLASS__, 'tab_tests' ),
            );
            $tabs = apply_filters( 'sn_tabs', $tabs );
            ?>
	<div class="wrap">
		<?php 
            $topbar = '<img src="' . WF_SN_PLUGIN_URL . 'images/sn-logo.svg' . '" height="28" alt="' . __( 'Visit wpsecurityninja.com', WF_SN_TEXT_DOMAIN ) . '" class="logoleft"><h1>Security Ninja <span>v.' . self::get_plugin_version() . '</span></h1>';
            echo  $topbar ;
            ?>
		<div class="secnin_content_wrapper">
			<div class="secnin_content_cell" id="secnin_content_top">
				<div class="nav-tab-wrapper" id="wf-sn-tabs">
					<?php 
            foreach ( $tabs as $tab ) {
                $class = 'nav-tab ' . $tab['class'];
                if ( $tab['id'] == 'sn_tests' ) {
                    $class .= ' nav-tab-active';
                }
                if ( !empty($tab['label']) ) {
                    echo  '<a href="#' . $tab['id'] . '" class="' . $class . '" id="' . $tab['id'] . '-tab">' . $tab['label'] . '</a>' ;
                }
            }
            // foreach
            ?>
	</div>
	<div id="sn_tabscont">
		<?php 
            foreach ( $tabs as $tab ) {
                
                if ( !empty($tab['callback']) ) {
                    $class = 'wf-sn-tab';
                    if ( $tab['id'] == 'sn_tests' ) {
                        $class .= ' active';
                    }
                    echo  '<div id="' . $tab['id'] . '" class="' . $class . '">' ;
                    call_user_func( $tab['callback'] );
                    echo  '</div>' ;
                }
            
            }
            // foreach
            ?>
	</div><!-- #sn_tabscont -->
	<?php 
            include_once 'misc/sidebar.php';
            ?>
</div><!-- #secnin_content_top -->
</div><!-- .secnin_content_wrapper -->
<?php 
            
            if ( function_exists( 'secnin_fs' ) ) {
                global  $secnin_fs ;
                $helpscoutbeacon = '';
                // Only loads beacon for registered users
                if ( $secnin_fs->is_registered() ) {
                    $helpscoutbeacon = '<script type="text/javascript">!function(e,t,n){function a(){var e=t.getElementsByTagName("script")[0],n=t.createElement("script");n.type="text/javascript",n.async=!0,n.src="https://beacon-v2.helpscout.net",e.parentNode.insertBefore(n,e)}if(e.Beacon=n=function(t,n,a){e.Beacon.readyQueue.push({method:t,options:n,data:a})},n.readyQueue=[],"complete"===t.readyState)return a();e.attachEvent?e.attachEvent("onload",a):e.addEventListener("load",a,!1)}(window,document,window.Beacon||function(){});</script>
		<script type="text/javascript">window.Beacon("init", "e7c8f66b-8c2f-472b-bc97-4a408d362ee0")</script>';
                }
                echo  $helpscoutbeacon ;
            }
            
            echo  '</div>' ;
        }
        
        // main_page
        /**
         * Compares two array values by for usort()
         * @param  [type] $a [description]
         * @param  [type] $b [description]
         * @return [type]    [description]
         */
        static function cmpStatusScore( $a, $b )
        {
            if ( $a == $b ) {
                return 0;
            }
            return ( $a['status'] < $b['status'] ? -1 : 1 );
        }
        
        /** Outputs warnings about other plugins or configurations */
        static function show_sec_tests_warnings()
        {
            // display warning if Wordfence plugin is active
            // Lars - Commented out since we do not have the bruteforce check currently. Future version will check in background.
            /*
            if (defined('WORDFENCE_VERSION') && WORDFENCE_VERSION) {
            	echo '<div class="secnotice secnotice-warning"><p>'.__("Please deactivate Wordfence before running Security Ninja tests. Some tests are detected as site attacks by Wordfence and hence can't be performed properly.", WF_SN_TEXT_DOMAIN).'</p><p>'.__("Activate Wordfence again once you're done testing.", WF_SN_TEXT_DOMAIN).'</p></div>';
            }
            */
            // Warning for CleanTalk
            if ( defined( 'APBCT_PLUGIN_BASE_NAME' ) && APBCT_PLUGIN_BASE_NAME == 'cleantalk-spam-protect/cleantalk.php' ) {
                echo  '<div class="secnotice secnotice-warning"><p>' . __( "Anti-Spam by CleanTalk plugin is activated.", WF_SN_TEXT_DOMAIN ) . '</p><p>' . __( "We have had user reports that this can prevent some tests from running. If you have problems, deactivate CleanTalk while running tests and remember to reactivate afterwards.", WF_SN_TEXT_DOMAIN ) . '</p></div>' ;
            }
        }
        
        // display tests table
        static function tab_tests()
        {
            // get test results from cache
            $tests = get_option( WF_SN_RESULTS_KEY );
            echo  '<div class="submit-test-container">' ;
            ?>

	<h3><?php 
            _e( 'Test your security', WF_SN_TEXT_DOMAIN );
            ?></h3>

	<?php 
            self::show_sec_tests_warnings();
            ?>

	<?php 
            echo  '<input type="submit" value="' . __( 'Analyze Site', WF_SN_TEXT_DOMAIN ) . '" id="run-tests" class="button button-primary button-hero" name="Submit" />' ;
            ?>

	<div class="testresults">
		<?php 
            
            if ( !empty($tests['last_run']) ) {
                ?>
			<p><?php 
                printf( esc_html__( 'Tests were last run on %s. It took %s seconds to run them. ', WF_SN_TEXT_DOMAIN ), date( get_option( 'date_format' ) . ' @ ' . get_option( 'time_format' ), $tests['last_run'] ), number_format( $tests['run_time'], 1 ) );
                ?>
		</p>
		<?php 
                $bad = $warning = $good = $score = $total = 0;
                foreach ( $tests['test'] as $test_details ) {
                    $total += $test_details['score'];
                    
                    if ( $test_details['status'] == 10 ) {
                        $good++;
                        $score += $test_details['score'];
                    } elseif ( $test_details['status'] == 0 ) {
                        $bad++;
                    } else {
                        $warning++;
                    }
                
                }
                $score = round( $score / $total * 100 );
                ?>
		<h3><?php 
                _e( 'Overall score', WF_SN_TEXT_DOMAIN );
                ?> - <?php 
                echo  count( $tests['test'] ) . ' tests' ;
                ?></h3>
		<?php 
                echo  '<div id="counters">' ;
                
                if ( $good == 1 ) {
                    echo  '<span class="good">' . $good . '<br><i>' . __( 'test passed', WF_SN_TEXT_DOMAIN ) . '</i></span>' ;
                } else {
                    echo  '<span class="good">' . $good . '<br><i>' . __( 'tests passed', WF_SN_TEXT_DOMAIN ) . '</i></span>' ;
                }
                
                
                if ( $warning == 1 ) {
                    echo  '<span class="warning">' . $warning . '<br><i>' . __( 'tests have warnings', WF_SN_TEXT_DOMAIN ) . '</i></span>' ;
                } else {
                    echo  '<span class="warning">' . $warning . '<br><i>' . __( 'tests have warnings', WF_SN_TEXT_DOMAIN ) . '</i></span>' ;
                }
                
                
                if ( $bad == 1 ) {
                    echo  '<span class="bad">' . $bad . '<br><i>' . __( 'test have failed', WF_SN_TEXT_DOMAIN ) . '</i></span>' ;
                } else {
                    echo  '<span class="bad">' . $bad . '<br><i>' . __( 'tests have failed', WF_SN_TEXT_DOMAIN ) . '</i></span>' ;
                }
                
                echo  '<span class="score">' . $score . '%<br><i>' . __( 'overall site score', WF_SN_TEXT_DOMAIN ) . '</i></span>' ;
                echo  '</div>' ;
            } else {
                //if ( (isset($tests['last_run'])) && (empty($tests['last_run'])) ) {
                ?>
		<h3><?php 
                _e( 'The security tests have not run', WF_SN_TEXT_DOMAIN );
                ?></h3>
		<p class="description"><?php 
                _e( 'Click "Analyze Site" to check for these security vulnerabilities:', WF_SN_TEXT_DOMAIN );
                ?></p>
		<?php 
                
                if ( is_array( wf_sn_tests::return_security_tests() ) ) {
                    ?>
			<ul class="security-test-list">
				<?php 
                    foreach ( wf_sn_tests::return_security_tests() as $test_name => $test ) {
                        ?>
					<li><?php 
                        echo  $test['title'] ;
                        ?></li>
					<?php 
                    }
                    ?>
			</ul>
			<?php 
                }
            
            }
            
            ?>
</div><!-- .testresults -->

</div>

<?php 
            $out = '';
            
            if ( !empty($tests['last_run']) ) {
                $out .= '<table class="wp-list-table widefat striped" cellspacing="0" id="security-ninja">';
                $out .= '<thead><tr>';
                //	$out .= '<th class="sn-status">'.__('Status', WF_SN_TEXT_DOMAIN).'</th>';
                $out .= '<th class="column-primary">' . __( 'Security Test', WF_SN_TEXT_DOMAIN ) . '</th>';
                //	$out .= '<th>'.__('Results', WF_SN_TEXT_DOMAIN).'</th>';
                $out .= '<th>&nbsp;</th>';
                $out .= '</tr></thead>';
                $out .= '<tbody>';
                /*
                $sortedtests = $tests['test'];
                
                
                //usort($sortedtests, array(__CLASS__, 'cmpStatusScore') );
                
                
                $tests['test'] = $sortedtests;
                */
                
                if ( is_array( $tests['test'] ) ) {
                    // test Results
                    foreach ( $tests['test'] as $test_name => $details ) {
                        if ( substr( $test_name, 0, 3 ) == 'ad_' || $test_name[0] == '_' ) {
                            continue;
                        }
                        $out .= '<tr class="wf-sn-test-row-status-' . $details['status'] . '">';
                        $outlabel = '';
                        
                        if ( $details['status'] == 0 ) {
                            $outlabel = '<span class="wf-sn-label sn-error">Failed</span>';
                        } elseif ( $details['status'] == 5 ) {
                            $outlabel = '<span class="wf-sn-label sn-warning">Warning</span>';
                        } elseif ( $details['status'] == 10 ) {
                            $outlabel = '<span class="wf-sn-label sn-success">Passed</span>';
                        }
                        
                        //	$out .= '<td>'.print_r($details["status"],true).'</td>';
                        // <td class="sn-status column-primary">' . wf_sn::status($details['status']) . '</td>
                        $out .= '<td class="column-primary" data-colname="Test">' . $outlabel . ' <span class="wf-sn-test-title">' . $details['title'] . '</span>';
                        if ( $details['status'] != 10 ) {
                            // only add details if failed or warning
                            $out .= '<span class="sn-result-details">' . $details['msg'] . '</span>';
                        }
                        $out .= '<button type="button" class="toggle-row">
		<span class="screen-reader-text">show details</span>
		</button></td>';
                        // 	<td>' . $details['msg'] . '</td>';
                        
                        if ( class_exists( 'wf_sn_af_fix_' . $test_name ) && $details['status'] != 10 ) {
                            $details_label = __( 'Details &amp; Fixer', WF_SN_TEXT_DOMAIN );
                        } else {
                            $details_label = __( 'Details', WF_SN_TEXT_DOMAIN );
                        }
                        
                        //		$out .= '<td class="sn-details"><a data-test-status="' . $details['status'] . '" data-test-id="' . $test_name . '" href="#' . $test_name . '" class="button action">' . $details_label . '</a></td>';
                        $out .= '<td class="sn-details"><a data-test-status="' . $details['status'] . '" data-test-id="' . $test_name . '" href="#' . $test_name . '" class="button action">' . $details_label . '</a></td>';
                        $out .= '</tr>';
                    }
                    // foreach ($tests)
                } else {
                    // no test results
                    $out .= '<tr>
				<td colspan="2">' . __( 'No test results are available. Click "Analyze Site" to run tests now.', WF_SN_TEXT_DOMAIN ) . '</td>
				</tr>';
                }
                
                // if tests
                $out .= '</tbody>';
                $out .= '<tfoot><tr>';
                $out .= '<th class="column-primary">' . __( 'Security Test', WF_SN_TEXT_DOMAIN ) . '</th>';
                $out .= '<th>&nbsp;</th>';
                $out .= '</tr></tfoot>';
                $out .= '</table>';
            }
            
            // if $results
            $out = apply_filters( 'sn_tests_table', $out, $tests );
            echo  $out ;
            ?>
		<p><?php 
            _e( "Although these tests cover years of best practices in security, getting all test green does not guarantee your site will not get hacked. Likewise, having them all red does not mean you will get hacked.", WF_SN_TEXT_DOMAIN );
            ?></p>
		<p><?php 
            _e( "Please read each test's detailed information to see if it represents a real security issue for your site.", WF_SN_TEXT_DOMAIN );
            ?></p>
		<?php 
        }
        
        // tab_tests
        // Runs the tests
        static function run_tests( $return = false )
        {
            if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
                check_ajax_referer( 'wf_sn_run_tests' );
            }
            // if( ! current_user_can( 'moderate_comments' ) ) {
            // 	return;
            // }
            $step = ( isset( $_POST['step'] ) ? absint( $_POST['step'] ) : 1 );
            if ( 1 == $step ) {
                self::timerstart( 'wf_sn_run_tests' );
            }
            if ( !$step ) {
                $step = 0;
            }
            $step++;
            $json_response = array();
            if ( $step ) {
                $json_response['step'] = $step;
            }
            $security_tests = wf_sn_tests::return_security_tests();
            
            if ( $security_tests ) {
                $totaltests = count( $security_tests );
                $json_response['totaltests'] = $totaltests;
            }
            
            // https://pippinsplugins.com/batch-processing-for-big-data/
            @set_time_limit( WF_SN_MAX_EXEC_SEC );
            $loop_count = 1;
            $start_time = microtime( true );
            $test_description['last_run'] = current_time( 'timestamp' );
            
            if ( is_array( $security_tests ) ) {
                foreach ( $security_tests as $test_name => $test ) {
                    if ( $test_name[0] == '_' || in_array( $test_name, self::$skip_tests ) || substr( $test_name, 0, 3 ) == 'ad_' ) {
                        continue;
                    }
                    // If this is the one to be tested ...
                    
                    if ( $step == $loop_count ) {
                        $response = wf_sn_tests::$test_name();
                        $json_response['last_test'] = $test['title'];
                        if ( isset( $response['status'] ) ) {
                            $json_response['last_status'] = $response['status'];
                        }
                        $json_response['last_score'] = $test['score'];
                        // allow overwriting with function response
                        if ( isset( $response['msg_bad'] ) ) {
                            $test['msg_bad'] = $response['msg_bad'];
                        }
                        if ( isset( $response['msg_ok'] ) ) {
                            $test['msg_ok'] = $response['msg_ok'];
                        }
                        if ( isset( $response['msg_warning'] ) ) {
                            $test['msg_warning'] = $response['msg_warning'];
                        }
                        if ( !isset( $response['msg'] ) ) {
                            $response['msg'] = '';
                        }
                        
                        if ( $response['status'] == 10 ) {
                            $json_response['last_msg'] = sprintf( $test['msg_ok'], $response['msg'] );
                        } elseif ( $response['status'] == 0 ) {
                            $json_response['last_msg'] = sprintf( $test['msg_bad'], $response['msg'] );
                        } else {
                            $json_response['last_msg'] = sprintf( $test['msg_warning'], $response['msg'] );
                        }
                        
                        // Updates the results
                        $resultssofar = get_option( WF_SN_RESULTS_KEY );
                        $resultssofar['test'][$test_name] = array(
                            'title'  => $test['title'],
                            'status' => $response['status'],
                            'score'  => $test['score'],
                            'msg'    => $json_response['last_msg'],
                        );
                        // A way to add details
                        if ( isset( $response['details'] ) ) {
                            $resultssofar['test'][$test_name]['details'] = $response['details'];
                        }
                        // No more tests - let us stop
                        
                        if ( $step >= $totaltests ) {
                            $json_response['step'] = 'done';
                            $resultssofar['last_run'] = current_time( 'timestamp' );
                            $stoptime = self::timerstop( 'wf_sn_run_tests' );
                            if ( $stoptime ) {
                                $resultssofar['run_time'] = $stoptime;
                            }
                            do_action( 'security_ninja_done_testing', $test_description, $resultssofar['run_time'] );
                        }
                        
                        update_option( WF_SN_RESULTS_KEY, $resultssofar );
                        wp_send_json_success( $json_response );
                    } else {
                        //	wp_send_json_error($json_response);
                    }
                    
                    $loop_count++;
                }
                // foreach
            }
            
            $pointers = get_option( WF_SN_POINTERS_KEY );
            
            if ( !empty($pointers['welcome']) ) {
                unset( $pointers['welcome'] );
                update_option( WF_SN_POINTERS_KEY, $pointers );
            }
            
            
            if ( $return ) {
                $resultssofar = get_option( WF_SN_RESULTS_KEY );
                return $resultssofar;
            } else {
                //update_option(WF_SN_RESULTS_KEY, $test_description);
                wp_send_json_success( $json_response );
            }
        
        }
        
        // run_test
        /*
        RUNS ALL TESTS, not just one
        */
        static function run_all_tests( $return = false )
        {
            if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
                check_ajax_referer( 'wf_sn_run_tests' );
            }
            self::timerstart( 'wf_sn_run_all_tests' );
            $security_tests = wf_sn_tests::return_security_tests();
            $resultssofar = array();
            @set_time_limit( WF_SN_MAX_EXEC_SEC );
            $loop_count = 1;
            $resultssofar['last_run'] = current_time( 'timestamp' );
            
            if ( is_array( $security_tests ) ) {
                foreach ( $security_tests as $test_name => $test ) {
                    if ( $test_name[0] == '_' || in_array( $test_name, self::$skip_tests ) || substr( $test_name, 0, 3 ) == 'ad_' ) {
                        continue;
                    }
                    $response = wf_sn_tests::$test_name();
                    $json_response = array();
                    $json_response['last_test'] = $test['title'];
                    $json_response['last_status'] = $response['status'];
                    $json_response['last_score'] = $test['score'];
                    if ( !isset( $response['msg'] ) ) {
                        $response['msg'] = '';
                    }
                    // Setting appropriate message
                    
                    if ( $response['status'] == 10 ) {
                        $json_response['last_msg'] = sprintf( $test['msg_ok'], $response['msg'] );
                    } elseif ( $response['status'] == 0 ) {
                        $json_response['last_msg'] = sprintf( $test['msg_bad'], $response['msg'] );
                    } else {
                        $json_response['last_msg'] = sprintf( $test['msg_warning'], $response['msg'] );
                    }
                    
                    // Updates the results
                    $resultssofar['test'][$test_name] = array(
                        'title'  => $test['title'],
                        'status' => $response['status'],
                        'score'  => $test['score'],
                        'msg'    => $json_response['last_msg'],
                    );
                    $loop_count++;
                }
                // No more tests - let us stop
                $json_response['step'] = 'done';
                $resultssofar['last_run'] = current_time( 'timestamp' );
                $stoptime = self::timerstop( 'wf_sn_run_all_tests' );
                if ( $stoptime ) {
                    $resultssofar['run_time'] = $stoptime;
                }
                update_option( WF_SN_RESULTS_KEY, $resultssofar );
            }
            
            // her stopper det sjove?
            do_action( 'security_ninja_done_testing', 'Security Tests - Completed Scanning', $resultssofar['run_time'] );
            
            if ( $return ) {
                $resultssofar = get_option( WF_SN_RESULTS_KEY );
                return $resultssofar;
            } else {
                wp_send_json_success();
            }
        
        }
        
        // run_all_tests
        // convert status integer to button
        static function status( $int )
        {
            
            if ( $int == 0 ) {
                $string = '<span class="sn-error">' . __( 'Failed', WF_SN_TEXT_DOMAIN ) . '</span>';
            } elseif ( $int == 10 ) {
                $string = '<span class="sn-success">' . __( 'Passed', WF_SN_TEXT_DOMAIN ) . '</span>';
            } else {
                $string = '<span class="sn-warning">' . __( 'Warning', WF_SN_TEXT_DOMAIN ) . '</span>';
            }
            
            return $string;
        }
        
        // status
        // reset pointers on activation and save some info
        static function activate()
        {
            $pointers = array();
            $pointers['welcome'] = array(
                'target'  => '#menu-tools',
                'edge'    => 'left',
                'align'   => 'right',
                'content' => __( 'Thank you for installing Security Ninja! Please open <a href="admin.php?page=wf-sn">Security Ninja</a> to analyze &amp; protect your site.', WF_SN_TEXT_DOMAIN ),
            );
            // @i8n
            update_option( WF_SN_POINTERS_KEY, $pointers );
            $options = self::get_options();
            
            if ( empty($options['first_version']) || empty($options['first_install']) ) {
                $options['first_version'] = wf_sn::get_plugin_version();
                $options['first_install'] = current_time( 'timestamp' );
                update_option( WF_SN_OPTIONS_KEY, $options );
            }
        
        }
        
        // activate
        // clean-up when deactivated
        static function deactivate()
        {
            delete_option( WF_SN_RESULTS_KEY );
            delete_option( WF_SN_POINTERS_KEY );
        }
        
        // clean-up when uninstalled
        static function uninstall()
        {
            delete_option( WF_SN_RESULTS_KEY );
            delete_option( WF_SN_OPTIONS_KEY );
            delete_option( WF_SN_POINTERS_KEY );
            delete_option( WF_SN_ACTIVE_PLUGINS );
            delete_option( WF_SN_REVIEW_NOTICE_KEY );
        }
    
    }
    // wf_sn class
    // ... Your plugin's main file logic ...
}

// hook everything up
register_activation_hook( __FILE__, array( 'WF_SN', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'WF_SN', 'deactivate' ) );
register_uninstall_hook( __FILE__, array( 'WF_SN', 'uninstall' ) );
add_action( 'init', array( 'WF_SN', 'init' ) );
add_action( 'plugins_loaded', array( 'WF_SN', 'plugins_loaded' ) );